/*
 * Decompiled with CFR 0.152.
 */
package org.apache.spark.network.shuffle;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.databind.ObjectMapper;
import java.io.File;
import java.io.FilenameFilter;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.util.Iterator;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Executor;
import java.util.concurrent.Executors;
import java.util.stream.Collectors;
import org.apache.commons.lang3.builder.ToStringBuilder;
import org.apache.commons.lang3.builder.ToStringStyle;
import org.apache.commons.lang3.tuple.Pair;
import org.apache.spark.internal.LogKey;
import org.apache.spark.internal.LogKeys;
import org.apache.spark.internal.MDC;
import org.apache.spark.internal.SparkLogger;
import org.apache.spark.internal.SparkLoggerFactory;
import org.apache.spark.network.buffer.FileSegmentManagedBuffer;
import org.apache.spark.network.buffer.ManagedBuffer;
import org.apache.spark.network.shuffle.AppsWithRecoveryDisabled;
import org.apache.spark.network.shuffle.ExecutorDiskUtils;
import org.apache.spark.network.shuffle.ShuffleIndexInformation;
import org.apache.spark.network.shuffle.ShuffleIndexRecord;
import org.apache.spark.network.shuffle.checksum.Cause;
import org.apache.spark.network.shuffle.checksum.ShuffleChecksumHelper;
import org.apache.spark.network.shuffle.protocol.ExecutorShuffleInfo;
import org.apache.spark.network.shuffledb.DB;
import org.apache.spark.network.shuffledb.DBBackend;
import org.apache.spark.network.shuffledb.DBIterator;
import org.apache.spark.network.shuffledb.StoreVersion;
import org.apache.spark.network.util.DBProvider;
import org.apache.spark.network.util.JavaUtils;
import org.apache.spark.network.util.NettyUtils;
import org.apache.spark.network.util.TransportConf;
import org.sparkproject.guava.annotations.VisibleForTesting;
import org.sparkproject.guava.cache.CacheBuilder;
import org.sparkproject.guava.cache.CacheLoader;
import org.sparkproject.guava.cache.LoadingCache;
import org.sparkproject.guava.collect.Maps;

public class ExternalShuffleBlockResolver {
    private static final SparkLogger logger = SparkLoggerFactory.getLogger(ExternalShuffleBlockResolver.class);
    private static final ObjectMapper mapper = new ObjectMapper();
    private static final String APP_KEY_PREFIX = "AppExecShuffleInfo";
    private static final StoreVersion CURRENT_VERSION = new StoreVersion(1, 0);
    @VisibleForTesting
    final ConcurrentMap<AppExecId, ExecutorShuffleInfo> executors;
    private final LoadingCache<String, ShuffleIndexInformation> shuffleIndexCache;
    private final Executor directoryCleaner;
    private final TransportConf conf;
    private final boolean rddFetchEnabled;
    @VisibleForTesting
    final File registeredExecutorFile;
    @VisibleForTesting
    final DB db;

    public ExternalShuffleBlockResolver(TransportConf conf, File registeredExecutorFile) throws IOException {
        this(conf, registeredExecutorFile, Executors.newSingleThreadExecutor(NettyUtils.createThreadFactory((String)"spark-shuffle-directory-cleaner")));
    }

    @VisibleForTesting
    ExternalShuffleBlockResolver(TransportConf conf, File registeredExecutorFile, Executor directoryCleaner) throws IOException {
        this.conf = conf;
        this.rddFetchEnabled = Boolean.parseBoolean(conf.get("spark.shuffle.service.fetch.rdd.enabled", "false"));
        this.registeredExecutorFile = registeredExecutorFile;
        String indexCacheSize = conf.get("spark.shuffle.service.index.cache.size", "100m");
        CacheLoader<String, ShuffleIndexInformation> indexCacheLoader = new CacheLoader<String, ShuffleIndexInformation>(){

            public ShuffleIndexInformation load(String filePath) throws IOException {
                return new ShuffleIndexInformation(filePath);
            }
        };
        this.shuffleIndexCache = CacheBuilder.newBuilder().maximumWeight(JavaUtils.byteStringAsBytes((String)indexCacheSize)).weigher((filePath, indexInfo) -> indexInfo.getRetainedMemorySize()).build((CacheLoader)indexCacheLoader);
        String dbBackendName = conf.get("spark.shuffle.service.db.backend", DBBackend.ROCKSDB.name());
        DBBackend dbBackend = DBBackend.byName((String)dbBackendName);
        this.db = DBProvider.initDB((DBBackend)dbBackend, (File)this.registeredExecutorFile, (StoreVersion)CURRENT_VERSION, (ObjectMapper)mapper);
        if (this.db != null) {
            logger.info("Use {} as the implementation of {}", new MDC[]{MDC.of((LogKey)LogKeys.SHUFFLE_DB_BACKEND_NAME$.MODULE$, (Object)dbBackend), MDC.of((LogKey)LogKeys.SHUFFLE_DB_BACKEND_KEY$.MODULE$, (Object)"spark.shuffle.service.db.backend")});
            this.executors = ExternalShuffleBlockResolver.reloadRegisteredExecutors(this.db);
        } else {
            this.executors = Maps.newConcurrentMap();
        }
        this.directoryCleaner = directoryCleaner;
    }

    public int getRegisteredExecutorsSize() {
        return this.executors.size();
    }

    public void registerExecutor(String appId, String execId, ExecutorShuffleInfo executorInfo) {
        AppExecId fullId = new AppExecId(appId, execId);
        logger.info("Registered executor {} with {}", new MDC[]{MDC.of((LogKey)LogKeys.APP_EXECUTOR_ID$.MODULE$, (Object)fullId), MDC.of((LogKey)LogKeys.EXECUTOR_SHUFFLE_INFO$.MODULE$, (Object)executorInfo)});
        try {
            if (this.db != null && AppsWithRecoveryDisabled.isRecoveryEnabledForApp(appId)) {
                byte[] key = ExternalShuffleBlockResolver.dbAppExecKey(fullId);
                byte[] value = mapper.writeValueAsString((Object)executorInfo).getBytes(StandardCharsets.UTF_8);
                this.db.put(key, value);
            }
        }
        catch (Exception e) {
            logger.error("Error saving registered executors", (Throwable)e);
        }
        this.executors.put(fullId, executorInfo);
    }

    public ManagedBuffer getBlockData(String appId, String execId, int shuffleId, long mapId, int reduceId) {
        return this.getContinuousBlocksData(appId, execId, shuffleId, mapId, reduceId, reduceId + 1);
    }

    public ManagedBuffer getContinuousBlocksData(String appId, String execId, int shuffleId, long mapId, int startReduceId, int endReduceId) {
        ExecutorShuffleInfo executor = (ExecutorShuffleInfo)this.executors.get(new AppExecId(appId, execId));
        if (executor == null) {
            throw new RuntimeException(String.format("Executor is not registered (appId=%s, execId=%s)", appId, execId));
        }
        return this.getSortBasedShuffleBlockData(executor, shuffleId, mapId, startReduceId, endReduceId);
    }

    public ManagedBuffer getRddBlockData(String appId, String execId, int rddId, int splitIndex) {
        ExecutorShuffleInfo executor = (ExecutorShuffleInfo)this.executors.get(new AppExecId(appId, execId));
        if (executor == null) {
            throw new RuntimeException(String.format("Executor is not registered (appId=%s, execId=%s)", appId, execId));
        }
        return this.getDiskPersistedRddBlockData(executor, rddId, splitIndex);
    }

    public void applicationRemoved(String appId, boolean cleanupLocalDirs) {
        logger.info("Application {} removed, cleanupLocalDirs = {}", new MDC[]{MDC.of((LogKey)LogKeys.APP_ID$.MODULE$, (Object)appId), MDC.of((LogKey)LogKeys.CLEANUP_LOCAL_DIRS$.MODULE$, (Object)cleanupLocalDirs)});
        Iterator it = this.executors.entrySet().iterator();
        while (it.hasNext()) {
            Map.Entry entry = it.next();
            AppExecId fullId = (AppExecId)entry.getKey();
            ExecutorShuffleInfo executor = (ExecutorShuffleInfo)entry.getValue();
            if (!appId.equals(fullId.appId)) continue;
            it.remove();
            if (this.db != null && AppsWithRecoveryDisabled.isRecoveryEnabledForApp(fullId.appId)) {
                try {
                    this.db.delete(ExternalShuffleBlockResolver.dbAppExecKey(fullId));
                }
                catch (IOException e) {
                    logger.error("Error deleting {} from executor state db", (Throwable)e, new MDC[]{MDC.of((LogKey)LogKeys.APP_ID$.MODULE$, (Object)appId)});
                }
            }
            if (!cleanupLocalDirs) continue;
            logger.info("Cleaning up executor {}'s {} local dirs", new MDC[]{MDC.of((LogKey)LogKeys.APP_EXECUTOR_ID$.MODULE$, (Object)fullId), MDC.of((LogKey)LogKeys.NUM_LOCAL_DIRS$.MODULE$, (Object)executor.localDirs.length)});
            this.directoryCleaner.execute(() -> this.deleteExecutorDirs(executor.localDirs));
        }
    }

    public void executorRemoved(String executorId, String appId) {
        logger.info("Clean up non-shuffle and non-RDD files associated with the finished executor {}", new MDC[]{MDC.of((LogKey)LogKeys.EXECUTOR_ID$.MODULE$, (Object)executorId)});
        AppExecId fullId = new AppExecId(appId, executorId);
        ExecutorShuffleInfo executor = (ExecutorShuffleInfo)this.executors.get(fullId);
        if (executor == null) {
            logger.info("Executor is not registered (appId={}, execId={})", new MDC[]{MDC.of((LogKey)LogKeys.APP_ID$.MODULE$, (Object)appId), MDC.of((LogKey)LogKeys.EXECUTOR_ID$.MODULE$, (Object)executorId)});
        } else {
            logger.info("Cleaning up non-shuffle and non-RDD files in executor {}'s {} local dirs", new MDC[]{MDC.of((LogKey)LogKeys.APP_EXECUTOR_ID$.MODULE$, (Object)fullId), MDC.of((LogKey)LogKeys.NUM_LOCAL_DIRS$.MODULE$, (Object)executor.localDirs.length)});
            this.directoryCleaner.execute(() -> this.deleteNonShuffleServiceServedFiles(executor.localDirs));
        }
    }

    private void deleteExecutorDirs(String[] dirs) {
        for (String localDir : dirs) {
            try {
                JavaUtils.deleteRecursively((File)new File(localDir));
                logger.debug("Successfully cleaned up directory: {}", (Object)localDir);
            }
            catch (Exception e) {
                logger.error("Failed to delete directory: {}", (Throwable)e, new MDC[]{MDC.of((LogKey)LogKeys.PATH$.MODULE$, (Object)localDir)});
            }
        }
    }

    private void deleteNonShuffleServiceServedFiles(String[] dirs) {
        FilenameFilter filter = (dir, name) -> !name.endsWith(".index") && !name.endsWith(".data") && (!this.rddFetchEnabled || !name.startsWith("rdd_"));
        for (String localDir : dirs) {
            try {
                JavaUtils.deleteRecursively((File)new File(localDir), (FilenameFilter)filter);
                logger.debug("Successfully cleaned up files not served by shuffle service in directory: {}", (Object)localDir);
            }
            catch (Exception e) {
                logger.error("Failed to delete files not served by shuffle service in directory: {}", (Throwable)e, new MDC[]{MDC.of((LogKey)LogKeys.PATH$.MODULE$, (Object)localDir)});
            }
        }
    }

    private ManagedBuffer getSortBasedShuffleBlockData(ExecutorShuffleInfo executor, int shuffleId, long mapId, int startReduceId, int endReduceId) {
        String indexFilePath = ExecutorDiskUtils.getFilePath(executor.localDirs, executor.subDirsPerLocalDir, "shuffle_" + shuffleId + "_" + mapId + "_0.index");
        try {
            ShuffleIndexInformation shuffleIndexInformation = (ShuffleIndexInformation)this.shuffleIndexCache.get((Object)indexFilePath);
            ShuffleIndexRecord shuffleIndexRecord = shuffleIndexInformation.getIndex(startReduceId, endReduceId);
            return new FileSegmentManagedBuffer(this.conf, new File(ExecutorDiskUtils.getFilePath(executor.localDirs, executor.subDirsPerLocalDir, "shuffle_" + shuffleId + "_" + mapId + "_0.data")), shuffleIndexRecord.offset(), shuffleIndexRecord.length());
        }
        catch (ExecutionException e) {
            throw new RuntimeException("Failed to open file: " + indexFilePath, e);
        }
    }

    public ManagedBuffer getDiskPersistedRddBlockData(ExecutorShuffleInfo executor, int rddId, int splitIndex) {
        File file = new File(ExecutorDiskUtils.getFilePath(executor.localDirs, executor.subDirsPerLocalDir, "rdd_" + rddId + "_" + splitIndex));
        long fileLength = file.length();
        FileSegmentManagedBuffer res = null;
        if (file.exists()) {
            res = new FileSegmentManagedBuffer(this.conf, file, 0L, fileLength);
        }
        return res;
    }

    void close() {
        if (this.db != null) {
            try {
                this.db.close();
            }
            catch (IOException e) {
                logger.error("Exception closing RocksDB with registered executors", (Throwable)e);
            }
        }
    }

    public int removeBlocks(String appId, String execId, String[] blockIds) {
        ExecutorShuffleInfo executor = (ExecutorShuffleInfo)this.executors.get(new AppExecId(appId, execId));
        if (executor == null) {
            throw new RuntimeException(String.format("Executor is not registered (appId=%s, execId=%s)", appId, execId));
        }
        int numRemovedBlocks = 0;
        for (String blockId : blockIds) {
            File file = new File(ExecutorDiskUtils.getFilePath(executor.localDirs, executor.subDirsPerLocalDir, blockId));
            if (file.delete()) {
                ++numRemovedBlocks;
                continue;
            }
            logger.warn("Failed to delete block: {}", new MDC[]{MDC.of((LogKey)LogKeys.PATH$.MODULE$, (Object)file.getAbsolutePath())});
        }
        return numRemovedBlocks;
    }

    public Map<String, String[]> getLocalDirs(String appId, Set<String> execIds) {
        return execIds.stream().map(exec -> {
            ExecutorShuffleInfo info = (ExecutorShuffleInfo)this.executors.get(new AppExecId(appId, (String)exec));
            if (info == null) {
                throw new RuntimeException(String.format("Executor is not registered (appId=%s, execId=%s)", appId, exec));
            }
            return Pair.of((Object)exec, (Object)info.localDirs);
        }).collect(Collectors.toMap(Pair::getKey, Pair::getValue));
    }

    public Cause diagnoseShuffleBlockCorruption(String appId, String execId, int shuffleId, long mapId, int reduceId, long checksumByReader, String algorithm) {
        ExecutorShuffleInfo executor = (ExecutorShuffleInfo)this.executors.get(new AppExecId(appId, execId));
        String fileName = "shuffle_" + shuffleId + "_" + mapId + "_0.checksum." + algorithm;
        File checksumFile = new File(ExecutorDiskUtils.getFilePath(executor.localDirs, executor.subDirsPerLocalDir, fileName));
        ManagedBuffer data = this.getBlockData(appId, execId, shuffleId, mapId, reduceId);
        return ShuffleChecksumHelper.diagnoseCorruption(algorithm, checksumFile, reduceId, data, checksumByReader);
    }

    private static byte[] dbAppExecKey(AppExecId appExecId) throws IOException {
        String appExecJson = mapper.writeValueAsString((Object)appExecId);
        String key = "AppExecShuffleInfo;" + appExecJson;
        return key.getBytes(StandardCharsets.UTF_8);
    }

    private static AppExecId parseDbAppExecKey(String s) throws IOException {
        if (!s.startsWith(APP_KEY_PREFIX)) {
            throw new IllegalArgumentException("expected a string starting with AppExecShuffleInfo");
        }
        String json = s.substring(APP_KEY_PREFIX.length() + 1);
        AppExecId parsed = (AppExecId)mapper.readValue(json, AppExecId.class);
        return parsed;
    }

    @VisibleForTesting
    static ConcurrentMap<AppExecId, ExecutorShuffleInfo> reloadRegisteredExecutors(DB db) throws IOException {
        ConcurrentMap registeredExecutors = Maps.newConcurrentMap();
        if (db != null) {
            try (DBIterator itr = db.iterator();){
                itr.seek(APP_KEY_PREFIX.getBytes(StandardCharsets.UTF_8));
                while (itr.hasNext()) {
                    Map.Entry e = (Map.Entry)itr.next();
                    String key = new String((byte[])e.getKey(), StandardCharsets.UTF_8);
                    if (!key.startsWith(APP_KEY_PREFIX)) {
                        break;
                    }
                    AppExecId id = ExternalShuffleBlockResolver.parseDbAppExecKey(key);
                    logger.info("Reloading registered executors: {}", new MDC[]{MDC.of((LogKey)LogKeys.APP_EXECUTOR_ID$.MODULE$, (Object)id)});
                    ExecutorShuffleInfo shuffleInfo = (ExecutorShuffleInfo)mapper.readValue((byte[])e.getValue(), ExecutorShuffleInfo.class);
                    registeredExecutors.put(id, shuffleInfo);
                }
            }
        }
        return registeredExecutors;
    }

    public static class AppExecId {
        public final String appId;
        public final String execId;

        @JsonCreator
        public AppExecId(@JsonProperty(value="appId") String appId, @JsonProperty(value="execId") String execId) {
            this.appId = appId;
            this.execId = execId;
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            AppExecId appExecId = (AppExecId)o;
            return Objects.equals(this.appId, appExecId.appId) && Objects.equals(this.execId, appExecId.execId);
        }

        public int hashCode() {
            return Objects.hash(this.appId, this.execId);
        }

        public String toString() {
            return new ToStringBuilder((Object)this, ToStringStyle.SHORT_PREFIX_STYLE).append("appId", (Object)this.appId).append("execId", (Object)this.execId).toString();
        }
    }
}

