/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.phoenix.parse;

import java.sql.SQLException;
import org.apache.phoenix.compile.ColumnResolver;

/**
 * Node representing a bind variable in a SQL expression
 * @since 0.1
 */
public class BindParseNode extends NamedParseNode {
  private final int index;

  BindParseNode(String name) {
    super(name);
    index = Integer.parseInt(name);
  }

  public int getIndex() {
    return index - 1;
  }

  @Override
  public <T> T accept(ParseNodeVisitor<T> visitor) throws SQLException {
    return visitor.visit(this);
  }

  @Override
  public boolean isStateless() {
    return true;
  }

  @Override
  public int hashCode() {
    final int prime = 31;
    int result = super.hashCode();
    result = prime * result + index;
    return result;
  }

  @Override
  public boolean equals(Object obj) {
    if (this == obj) return true;
    if (!super.equals(obj)) return false;
    if (getClass() != obj.getClass()) return false;
    BindParseNode other = (BindParseNode) obj;
    if (index != other.index) return false;
    return true;
  }

  @Override
  public void toSQL(ColumnResolver resolver, StringBuilder buf) {
    buf.append(':');
    buf.append(index);
  }
}
