/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.kie.dmn.signavio.feel.runtime;

import java.math.BigDecimal;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.OffsetDateTime;
import java.time.ZoneId;
import java.time.ZoneOffset;
import java.time.ZonedDateTime;
import java.time.format.DateTimeFormatter;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;

import org.junit.jupiter.params.ParameterizedTest;
import org.junit.jupiter.params.provider.MethodSource;
import org.kie.dmn.api.feel.runtime.events.FEELEvent;

import static org.kie.dmn.api.feel.runtime.events.FEELEvent.Severity.ERROR;
import static org.kie.dmn.signavio.util.DynamicTypeUtils.entry;
import static org.kie.dmn.signavio.util.DynamicTypeUtils.mapOf;

public class FEELExtendedFunctionsTest extends ExtendedFunctionsBaseFEELTest {

    @ParameterizedTest
    @MethodSource("data")
    protected void instanceTest(String expression, Object result, FEELEvent.Severity severity) {
        expression(expression, result, severity);
    }

    private static Collection<Object[]> data() {
        final Object[][] cases = new Object[][] {
                { "string(\"Happy %.0fth birthday, Mr %s!\", 38, \"Doe\")", "Happy 38th birthday, Mr Doe!", null},
                { "all( true, true, true )", Boolean.TRUE , null},
                { "all([ true, true, true ])", Boolean.TRUE , null},
                { "all( true, true, false )", Boolean.FALSE , null},
                { "all([ false ])", Boolean.FALSE , null},
                { "any( false, true, false )", Boolean.TRUE , null},
                { "any([ false, true, false ])", Boolean.TRUE , null},
                { "any( false )", Boolean.FALSE , null},
                { "any([ false, false, false ])", Boolean.FALSE , null},
                { "now()", ZonedDateTime.class , null},
                { "today()", LocalDate.class, null },
                { "abs( -10 )", new BigDecimal( "10" ), null },
                { "abs( 20 )", new BigDecimal( "20" ), null },
                { "abs( null )", null, ERROR },
                { "round( 5.235, 2 )", new BigDecimal( "5.24" ), null },
                { "round( 5.345, 2 )", new BigDecimal( "5.34" ), null },
                { "round( 5.234 )", new BigDecimal( "5" ), null },
                { "round( null )", null, ERROR },
                { "round( null, 2 )", null, ERROR },
                { "round( 5.234, null )", new BigDecimal( "5" ), null },
                { "roundUp( 1.344, 2 )", new BigDecimal( "1.35" ), null },
                { "roundUp( 5.234 )", new BigDecimal( "6" ), null },
                { "roundUp( null, 2 )", null, ERROR },
                { "roundUp( 5.234, null )", new BigDecimal( "6" ), null },
                { "roundDown( 1.349, 2 )", new BigDecimal( "1.34" ), null },
                { "roundDown( 1.987 )",new BigDecimal( "1" ), null },
                { "roundDown( null, 2 )", null, ERROR },
                { "roundDown( 1.987, null )",new BigDecimal( "1" ), null },
                { "integer( 1.5 )", new BigDecimal( "1" ), null },
                { "integer( -1.5 )", new BigDecimal( "-1" ), null },
                { "integer( null )", null, ERROR },
                { "modulo( 4, 3 )", new BigDecimal( "1" ), null },
                { "modulo( null, 3 )", null, ERROR },
                { "modulo( 4, null )", null, ERROR },
                { "percent( 23 )", new BigDecimal( "0.23" ), null },
                { "percent( null )", null, ERROR },
                { "power( 2, 3 )", new BigDecimal( "8" ), null },
                { "power( 2, -3 )", new BigDecimal( "0.125" ), null },
                { "power( null, 3 )", null, ERROR },
                { "power( 2, null )", null, ERROR },
                { "product( [ 2, 3, 4 ] )", new BigDecimal( "24" ), null },
                { "product( 2, 3, 4 )", new BigDecimal( "24" ), null },
                { "product( 2 )", new BigDecimal( "2" ), null },
                { "product( [] )", new BigDecimal( "0" ), null },
                { "product( list: null )", null, ERROR },
                { "product( [\"a\", \"b\"] )", null, ERROR },
                { "product( \"a\", \"b\" )", null, ERROR },
                { "day( \"2017-07-02\" )", new BigDecimal( "2" ), null },
                { "day( \"2017-07-02T14:40:23\" )", new BigDecimal( "2" ), null },
                { "day( \"2017-07-02T14:40:23-06:00\" )", new BigDecimal( "2" ), null },
                { "day( \"2017-07-02T14:40:23Z\" )", new BigDecimal( "2" ), null },
                { "day( date( \"2017-07-02\" ) )", new BigDecimal( "2" ), null },
                { "day( date and time( \"2017-07-02T14:40:23-06:00\") )", new BigDecimal( "2" ), null },
                { "day( datestring: null )", null, ERROR },
                { "day( datestring: \"2017-07-12345\" )", null, ERROR },
                { "day( datetime: null )", null, ERROR },
                { "month( \"2017-07-02\" )", new BigDecimal( "7" ), null },
                { "month( \"2017-07-02T14:40:23\" )", new BigDecimal( "7" ), null },
                { "month( \"2017-07-02T14:40:23-06:00\" )", new BigDecimal( "7" ), null },
                { "month( \"2017-07-02T14:40:23Z\" )", new BigDecimal( "7" ), null },
                { "month( date( \"2017-07-02\" ) )", new BigDecimal( "7" ), null },
                { "month( date and time( \"2017-07-02T14:40:23-06:00\") )", new BigDecimal( "7" ), null },
                { "month( null )", null, ERROR },
                { "month( \"2017-07-02456\" )", null, ERROR },
                { "month( datestring: null )", null, ERROR },
                { "month( datetime: null )", null, ERROR },
                { "year( \"2017-07-02\" )", new BigDecimal( "2017" ), null },
                { "year( \"2017-07-02T14:40:23\" )", new BigDecimal( "2017" ), null },
                { "year( \"2017-07-02T14:40:23-06:00\" )", new BigDecimal( "2017" ), null },
                { "year( \"2017-07-02T14:40:23Z\" )", new BigDecimal( "2017" ), null },
                { "year( date( \"2017-07-02\" ) )", new BigDecimal( "2017" ), null },
                { "year( date and time( \"2017-07-02T14:40:23-06:00\") )", new BigDecimal( "2017" ), null },
                { "year( datestring: null )",null, ERROR },
                { "year( \"2017-07-023456\" )", null, ERROR },
                { "year( datetime: null )", null, ERROR },
                { "hour( \"2017-07-02T14:40:23\" )", new BigDecimal( "14" ), null },
                { "hour( \"2017-07-02T14:40:23-06:00\" )", new BigDecimal( "14" ), null },
                { "hour( \"2017-07-02T14:40:23Z\" )", new BigDecimal( "14" ), null },
                { "hour( \"14:40:23\" )", new BigDecimal( "14" ), null },
                { "hour( \"14:40:23+05:00\" )", new BigDecimal( "14" ), null },
                { "hour( time( \"14:40:23\" ) )", new BigDecimal( "14" ), null },
                { "hour( date and time( \"2017-07-02T14:40:23-06:00\") )", new BigDecimal( "14" ), null },
                { "hour( null )", null, ERROR },
                { "hour( \"14:40:23456\" )", null, ERROR },
                { "minute( \"2017-07-02T14:40:23\" )", new BigDecimal( "40" ), null },
                { "minute( \"2017-07-02T14:40:23-06:00\" )", new BigDecimal( "40" ), null },
                { "minute( \"2017-07-02T14:40:23Z\" )", new BigDecimal( "40" ), null },
                { "minute( \"14:40:23\" )", new BigDecimal( "40" ), null },
                { "minute( \"14:40:23+05:00\" )", new BigDecimal( "40" ), null },
                { "minute( time( \"14:40:23\" ) )", new BigDecimal( "40" ), null },
                { "minute( date and time( \"2017-07-02T14:40:23-06:00\") )", new BigDecimal( "40" ), null },
                { "minute( null )", null, ERROR },
                { "minute( \"14:40:23456\" )", null, ERROR },
                { "second( \"2017-07-02T14:40:23\" )", new BigDecimal( "23" ), null },
                { "second( \"2017-07-02T14:40:23-06:00\" )", new BigDecimal( "23" ), null },
                { "second( \"2017-07-02T14:40:23Z\" )", new BigDecimal( "23" ), null },
                { "second( \"14:40:23\" )", new BigDecimal( "23" ), null },
                { "second( \"14:40:23+05:00\" )", new BigDecimal( "23" ), null },
                { "second( time( \"14:40:23\" ) )", new BigDecimal( "23" ), null },
                { "second( date and time( \"2017-07-02T14:40:23-06:00\") )", new BigDecimal( "23" ), null },
                { "second( null )", null, ERROR },
                { "second( \"14:40:23456\" )", null, ERROR },
                { "yearDiff( date and time( \"2015-07-02T14:40:23-06:00\" ), date and time(\"2017-07-02T14:40:23-06:00\") )", new BigDecimal( "2" ), null },
                { "yearDiff( date and time( \"2015-06-02T14:40:23-06:00\" ), date and time(\"2017-07-02T14:40:23-06:00\") )", new BigDecimal( "2" ), null },
                { "yearDiff( date and time( \"2015-08-02T14:40:23-06:00\" ), date and time(\"2017-07-02T14:40:23-06:00\") )", new BigDecimal( "1" ), null },
                { "yearDiff( \"2015-07-02T14:40:23-06:00\", \"2017-07-02T14:40:23-06:00\" )", new BigDecimal( "2" ), null },
                { "yearDiff( \"2015-06-02T14:40:23-06:00\", \"2017-07-02T14:40:23-06:00\" )", new BigDecimal( "2" ), null },
                { "yearDiff( \"2015-08-02T14:40:23-06:00\", \"2017-07-02T14:40:23-06:00\" )", new BigDecimal( "1" ), null },
                { "yearDiff( null, date and time(\"2017-07-02T14:40:23-06:00\") )", null, ERROR },
                { "yearDiff( date and time( \"2015-07-02T14:40:23-06:00\" ), null )", null, ERROR },
                { "yearDiff( null, \"2017-07-02T14:40:23-06:00\" )",null, ERROR },
                { "yearDiff( \"2015-07-02T14:40:23-06:00\", null )", null, ERROR },
                { "yearDiff( \"2015-07-02T14:40:23456-06:00\", \"2017-07-02T14:40:23-06:00\" )", null, ERROR },
                { "yearDiff( \"2015-07-02T14:40:23-06:00\", \"2017-07-02T14:40:23456-06:00\" )", null, ERROR },
                { "dayAdd( date and time( \"2015-07-02T14:40:23-06:00\" ), 2 )", ZonedDateTime.of( 2015, 7, 4, 14, 40, 23, 0, ZoneOffset.ofHours( -6 ) ), null },
                { "dayAdd( date and time( \"2015-07-02T14:40:23-06:00\" ), -2 )", ZonedDateTime.of( 2015, 6, 30, 14, 40, 23, 0, ZoneOffset.ofHours( -6 ) ), null },
                { "dayAdd( \"2015-07-02T14:40:23-06:00\", 2 )", ZonedDateTime.of( 2015, 7, 4, 14, 40, 23, 0, ZoneOffset.ofHours( -6 ) ), null },
                { "dayAdd( \"2015-07-02T14:40:23-06:00\", -2 )", ZonedDateTime.of( 2015, 6, 30, 14, 40, 23, 0, ZoneOffset.ofHours( -6 ) ), null },
                { "dayAdd( date( \"2015-07-02\" ), 2 )", LocalDate.of( 2015, 7, 4 ), null },
                { "dayAdd( date( \"2015-07-02\" ), -2 )", LocalDate.of( 2015, 6, 30 ), null },
                { "dayAdd( \"2015-07-02\", 2 )", LocalDate.of( 2015, 7, 4 ), null },
                { "dayAdd( \"2015-07-02\", -2 )", LocalDate.of( 2015, 6, 30 ), null },
                { "dayAdd( datestring: null, days to add: -2 )", null, ERROR },
                { "dayAdd( datestring: \"2015-07-02\", days to add: null )", null, ERROR },
                { "dayAdd( \"2015-07-12345\", -2 )", null, ERROR },
                { "dayAdd( datetime: null, days to add: -2 )", null, ERROR },
                { "dayAdd( datetime: date(\"2015-07-02\"), days to add: null )", null, ERROR },
                { "dayDiff( date and time( \"2015-07-02T14:40:23-06:00\" ), date and time(\"2017-07-02T14:40:23-06:00\") )", new BigDecimal( "731" ), null },
                { "dayDiff( date and time( \"2015-06-02T14:40:23-06:00\" ), date and time(\"2017-07-02T14:40:23-06:00\") )", new BigDecimal( "761" ), null },
                { "dayDiff( date and time( \"2015-08-02T14:40:23-06:00\" ), date and time(\"2017-07-02T14:40:23-06:00\") )", new BigDecimal( "700" ), null },
                { "dayDiff( \"2015-07-02T14:40:23-06:00\", \"2017-07-02T14:40:23-06:00\" )", new BigDecimal( "731" ), null },
                { "dayDiff( \"2015-06-02T14:40:23-06:00\", \"2017-07-02T14:40:23-06:00\" )", new BigDecimal( "761" ), null },
                { "dayDiff( \"2015-08-02T14:40:23-06:00\", \"2017-07-02T14:40:23-06:00\" )", new BigDecimal( "700" ), null },
                { "dayDiff( date ( \"2015-08-02\" ), date( \"2017-07-02\" ) )", new BigDecimal( "700" ), null },
                { "dayDiff( date ( \"2015-08-02\" ), date and time(\"2017-07-02T14:40:23-06:00\") )", new BigDecimal( "700" ), null },
                { "dayDiff( date and time( \"2015-08-02T14:40:23-06:00\" ), date( \"2017-07-02\" ) )", new BigDecimal( "699" ), null },
                { "dayDiff( date( \"2015-08-02\" ), today() )", BigDecimal.class, null },
                { "dayDiff( date and time( \"2015-07-02T14:40:23-06:00\" ), today() )", BigDecimal.class, null },
                { "dayDiff( date( \"2015-08-02\" ), now() )", BigDecimal.class, null },
                { "dayDiff( date and time( \"2015-07-02T14:40:23-06:00\" ), now() )", BigDecimal.class, null },
                { "dayDiff( datetime1: null, datetime2: date( \"2017-07-02\" ) )", null, ERROR },
                { "dayDiff( datetime1: date( \"2017-07-02\" ), datetime2: null )", null, ERROR },
                { "dayDiff( datetime1: null, datetime2: \"2017-07-02\" )", null, ERROR },
                { "dayDiff( datetime1: \"2017-07-02\", datetime2: null )", null, ERROR },
                { "dayDiff( datetime1: \"2015-08-02\", datetime2: \"2017-07-12345\" )", null, ERROR },
                { "dayDiff( datetime1: \"2015-08-12345\", datetime2: \"2017-07-02\" )", null, ERROR },
                { "dateTime(\"2016-07-29T05:48:23\")", LocalDateTime.of( 2016, 7, 29, 5, 48, 23, 0 ) , null},
                { "dateTime( 2016, 7, 29, 5, 48, 23 )", LocalDateTime.of( 2016, 7, 29, 5, 48, 23, 0 ) , null},
                { "dateTime(\"2016-07-29T05:48:23Z\")", ZonedDateTime.of(2016, 7, 29, 5, 48, 23, 0, ZoneId.of("Z").normalized()) , null},
                { "dateTime( 2016, 7, 29, 5, 48, 23, -5 )", OffsetDateTime.of(2016, 7, 29, 5, 48, 23, 0, ZoneOffset.ofHours( -5 ) ) , null},
                { "dateTime(\"2016-07-29T05:48:23.765-05:00\")", DateTimeFormatter.ISO_DATE_TIME.parse( "2016-07-29T05:48:23.765-05:00", ZonedDateTime::from ) , null},
                { "dateTime(date(\"2016-07-29\"), time(\"05:48:23.765-05:00\") )", DateTimeFormatter.ISO_DATE_TIME.parse( "2016-07-29T05:48:23.765-05:00", ZonedDateTime::from ) , null},
                { "hourDiff( date and time( \"2015-07-02T14:40:23-06:00\" ), date and time(\"2017-07-02T14:40:23-06:00\") )", new BigDecimal( "17544" ), null },
                { "hourDiff( date and time( \"2015-06-02T14:40:23-06:00\" ), date and time(\"2017-07-02T14:40:23-06:00\") )", new BigDecimal( "18264" ), null },
                { "hourDiff( date and time( \"2015-08-02T14:40:23-06:00\" ), date and time(\"2017-07-02T14:40:23-06:00\") )", new BigDecimal( "16800" ), null },
                { "hourDiff( \"2015-07-02T14:40:23-06:00\", \"2017-07-02T14:40:23-06:00\" )", new BigDecimal( "17544" ), null },
                { "hourDiff( \"2015-06-02T14:40:23-06:00\", \"2017-07-02T14:40:23-06:00\" )", new BigDecimal( "18264" ), null },
                { "hourDiff( \"2015-08-02T14:40:23-06:00\", \"2017-07-02T14:40:23-06:00\" )", new BigDecimal( "16800" ), null },
                { "hourDiff( datetime1: null, datetime2: date and time( \"2017-07-02T14:40:23-06:00\" ) )", null, ERROR },
                { "hourDiff( datetime1: date and time( \"2015-07-02T14:40:23-06:00\" ), datetime2: null )", null, ERROR },
                { "hourDiff( datetime1: null, datetime2: \"2017-07-02T14:40:23-06:00\" )", null, ERROR },
                { "hourDiff( datetime1: \"2015-07-02T14:40:23-06:00\", datetime2: null )", null, ERROR },
                { "hourDiff( datetime1: \"2015-07-02T14:40:23-06:00\", datetime2: \"2017-07-02T14:40:23456-06:00\" )", null, ERROR },
                { "hourDiff( datetime1: \"2015-07-02T14:40:23456-06:00\", datetime2: \"2017-07-02T14:40:23-06:00\" )", null, ERROR },
                { "minutesDiff( date and time( \"2015-07-02T14:40:23-06:00\" ), date and time(\"2017-07-02T14:40:23-06:00\") )", new BigDecimal( "1052640" ), null },
                { "minutesDiff( date and time( \"2015-06-02T14:40:23-06:00\" ), date and time(\"2017-07-02T14:40:23-06:00\") )", new BigDecimal( "1095840" ), null },
                { "minutesDiff( date and time( \"2015-08-02T14:40:23-06:00\" ), date and time(\"2017-07-02T14:40:23-06:00\") )", new BigDecimal( "1008000" ), null },
                { "minutesDiff( \"2015-07-02T14:40:23-06:00\", \"2017-07-02T14:40:23-06:00\" )", new BigDecimal( "1052640" ), null },
                { "minutesDiff( \"2015-06-02T14:40:23-06:00\", \"2017-07-02T14:40:23-06:00\" )", new BigDecimal( "1095840" ), null },
                { "minutesDiff( \"2015-08-02T14:40:23-06:00\", \"2017-07-02T14:40:23-06:00\" )", new BigDecimal( "1008000" ), null },
                { "minutesDiff( datetime1: null, datetime2: date and time( \"2017-07-02T14:40:23-06:00\" ) )", null, ERROR },
                { "minutesDiff( datetime1: date and time( \"2015-07-02T14:40:23-06:00\" ), datetime2: null )", null, ERROR },
                { "minutesDiff( datetime1: null, datetime2: \"2017-07-02T14:40:23-06:00\" )", null, ERROR },
                { "minutesDiff( datetime1: \"2015-07-02T14:40:23-06:00\", datetime2: null )", null, ERROR },
                { "minutesDiff( datetime1: \"2015-07-02T14:40:23-06:00\", datetime2: \"2017-07-02T14:40:23456-06:00\" )", null, ERROR },
                { "minutesDiff( datetime1: \"2015-07-02T14:40:23456-06:00\", datetime2: \"2017-07-02T14:40:23-06:00\" )", null, ERROR },
                { "secondsDiff( date and time( \"2015-07-02T14:40:23-06:00\" ), date and time(\"2017-07-02T14:40:23-06:00\") )", new BigDecimal( "63158400" ), null },
                { "secondsDiff( date and time( \"2015-06-02T14:40:23-06:00\" ), date and time(\"2017-07-02T14:40:23-06:00\") )", new BigDecimal( "65750400" ), null },
                { "secondsDiff( date and time( \"2015-08-02T14:40:23-06:00\" ), date and time(\"2017-07-02T14:40:23-06:00\") )", new BigDecimal( "60480000" ), null },
                { "secondsDiff( \"2015-07-02T14:40:23-06:00\", \"2017-07-02T14:40:23-06:00\" )", new BigDecimal( "63158400" ), null },
                { "secondsDiff( \"2015-06-02T14:40:23-06:00\", \"2017-07-02T14:40:23-06:00\" )", new BigDecimal( "65750400" ), null },
                { "secondsDiff( \"2015-08-02T14:40:23-06:00\", \"2017-07-02T14:40:23-06:00\" )", new BigDecimal( "60480000" ), null },
                { "secondsDiff( datetime1: null, datetime2: date and time( \"2017-07-02T14:40:23-06:00\" ) )", null, ERROR },
                { "secondsDiff( datetime1: date and time( \"2015-07-02T14:40:23-06:00\" ), datetime2: null )", null, ERROR },
                { "secondsDiff( datetime1: null, datetime2: \"2017-07-02T14:40:23-06:00\" )", null, ERROR },
                { "secondsDiff( datetime1: \"2015-07-02T14:40:23-06:00\", datetime2: null )", null, ERROR },
                { "secondsDiff( datetime1: \"2015-07-02T14:40:23-06:00\", datetime2: \"2017-07-02T14:40:23456-06:00\" )", null, ERROR },
                { "secondsDiff( datetime1: \"2015-07-02T14:40:23456-06:00\", datetime2: \"2017-07-02T14:40:23-06:00\" )", null, ERROR },
                { "monthAdd( date and time( \"2015-07-02T14:40:23-06:00\" ), 2 )", ZonedDateTime.of( 2015, 9, 2, 14, 40, 23, 0, ZoneOffset.ofHours( -6 ) ), null },
                { "monthAdd( date and time( \"2015-07-02T14:40:23-06:00\" ), -2 )", ZonedDateTime.of( 2015, 5, 2, 14, 40, 23, 0, ZoneOffset.ofHours( -6 ) ), null },
                { "monthAdd( \"2015-07-02T14:40:23-06:00\", 2 )", ZonedDateTime.of( 2015, 9, 2, 14, 40, 23, 0, ZoneOffset.ofHours( -6 ) ), null },
                { "monthAdd( \"2015-07-02T14:40:23-06:00\", -2 )", ZonedDateTime.of( 2015, 5, 2, 14, 40, 23, 0, ZoneOffset.ofHours( -6 ) ), null },
                { "monthAdd( date( \"2015-07-02\" ), 2 )", LocalDate.of( 2015, 9, 2 ), null },
                { "monthAdd( date( \"2015-07-02\" ), -2 )", LocalDate.of( 2015, 5, 2 ), null },
                { "monthAdd( \"2015-07-02\", 2 )", LocalDate.of( 2015, 9, 2 ), null },
                { "monthAdd( \"2015-07-02\", -2 )", LocalDate.of( 2015, 5, 2 ), null },
                { "monthAdd( datestring: null, months to add: 2 )", null, ERROR },
                { "monthAdd( datestring: \"2015-07-02\", months to add: null )", null, ERROR },
                { "monthAdd( datestring: \"2015-07-023456\", months to add: null )", null, ERROR },
                { "monthAdd( datetime: null, months to add: 2 )", null, ERROR },
                { "monthAdd( datetime: date(\"2015-07-02\"), months to add: null )", null, ERROR },
                { "monthDiff( date and time( \"2015-07-02T14:40:23-06:00\" ), date and time(\"2017-07-02T14:40:23-06:00\") )", new BigDecimal( "24" ), null },
                { "monthDiff( date and time( \"2015-06-02T14:40:23-06:00\" ), date and time(\"2017-07-02T14:40:23-06:00\") )", new BigDecimal( "25" ), null },
                { "monthDiff( date and time( \"2015-08-02T14:40:23-06:00\" ), date and time(\"2017-07-02T14:40:23-06:00\") )", new BigDecimal( "23" ), null },
                { "monthDiff( \"2015-07-02T14:40:23-06:00\", \"2017-07-02T14:40:23-06:00\" )", new BigDecimal( "24" ), null },
                { "monthDiff( \"2015-06-02T14:40:23-06:00\", \"2017-07-02T14:40:23-06:00\" )", new BigDecimal( "25" ), null },
                { "monthDiff( \"2015-08-02T14:40:23-06:00\", \"2017-07-02T14:40:23-06:00\" )", new BigDecimal( "23" ), null },
                { "monthDiff( null, date and time(\"2017-07-02T14:40:23-06:00\") )", null, ERROR },
                { "monthDiff( date and time( \"2015-07-02T14:40:23-06:00\" ), null )", null, ERROR },
                { "monthDiff( null, \"2017-07-02T14:40:23-06:00\" )", null, ERROR },
                { "monthDiff( \"2015-07-02T14:40:23-06:00\", null )", null, ERROR },
                { "monthDiff( \"2015-07-02T14:40:23456-06:00\", \"2017-07-02T14:40:23-06:00\" )", null, ERROR },
                { "monthDiff( \"2015-07-02T14:40:23-06:00\", \"2017-07-02T14:40:23456-06:00\" )", null, ERROR },
                { "weekday( \"2016-02-09\" )", new BigDecimal( "2" ), null },
                { "weekday( \"2016-02-09T14:40:23\" )", new BigDecimal( "2" ), null },
                { "weekday( \"2016-02-09T14:40:23-06:00\" )", new BigDecimal( "2" ), null },
                { "weekday( \"2016-02-09T14:40:23Z\" )", new BigDecimal( "2" ), null },
                { "weekday( date( \"2016-02-09\" ) )", new BigDecimal( "2" ), null },
                { "weekday( datestring: null )", null, ERROR },
                { "weekday( datestring: \"2016-02-09T14:40:23456\" )", null, ERROR },
                { "weekday( datetime: null )", null, ERROR },
                { "weekday( date and time( \"2016-02-09T14:40:23-06:00\") )", new BigDecimal( "2" ), null },
                { "yearAdd( date and time( \"2015-07-02T14:40:23-06:00\" ), 2 )", ZonedDateTime.of( 2017, 7, 2, 14, 40, 23, 0, ZoneOffset.ofHours( -6 ) ), null },
                { "yearAdd( date and time( \"2015-07-02T14:40:23-06:00\" ), -2 )", ZonedDateTime.of( 2013, 7, 2, 14, 40, 23, 0, ZoneOffset.ofHours( -6 ) ), null },
                { "yearAdd( \"2015-07-02T14:40:23-06:00\", 2 )", ZonedDateTime.of( 2017, 7, 2, 14, 40, 23, 0, ZoneOffset.ofHours( -6 ) ), null },
                { "yearAdd( \"2015-07-02T14:40:23-06:00\", -2 )", ZonedDateTime.of( 2013, 7, 2, 14, 40, 23, 0, ZoneOffset.ofHours( -6 ) ), null },
                { "yearAdd( date( \"2015-07-02\" ), 2 )", LocalDate.of( 2017, 7, 2 ), null },
                { "yearAdd( date( \"2015-07-02\" ), -2 )", LocalDate.of( 2013, 7, 2 ), null },
                { "yearAdd( \"2015-07-02\", 2 )", LocalDate.of( 2017, 7, 2 ), null },
                { "yearAdd( \"2015-07-02\", -2 )", LocalDate.of( 2013, 7, 2 ), null },
                { "yearAdd( datestring: null, years to add: -2 )", null, ERROR },
                { "yearAdd( \"2015-07-02\", null )", null, ERROR },
                { "yearAdd( \"2015-07-023456\", -2 )", null, ERROR },
                { "yearAdd( datetime: null, years to add: -2 )", null, ERROR },
                { "yearAdd( date(\"2015-07-02\"), null )", null, ERROR },

                {"append([2.5, 5.8, 4.3], 6.7)", Arrays.asList(new BigDecimal("2.5"), new BigDecimal("5.8"), new BigDecimal("4.3"), new BigDecimal("6.7")), null},
                {"appendAll([2.5, 5.8, 4.3], [2.1, 3.5, 7.4])", Arrays.asList(new BigDecimal("2.5"), new BigDecimal("5.8"), new BigDecimal("4.3"), new BigDecimal("2.1"), new BigDecimal("3.5"), new BigDecimal("7.4")), null},
                {"notContainsAny([\"item1\", \"item2\"], [\"item2\", \"item3\"])", false, null},
                {"notContainsAny([\"item1\", \"item2\"], [\"item3\"])", true, null},
                {"notContainsAny([\"item1\", \"item2\"], [])", true, null},
                {"notContainsAny([\"item1\", \"item2\"], null)", true, null},
                {"notContainsAny(null, [])", null, ERROR},
                {"containsOnly([\"item1\", \"item2\"], [\"item2\", \"item3\"])", false, null},
                {"containsOnly([\"item1\", \"item2\"], [\"item2\"])", false, null},
                {"containsOnly([\"item1\", \"item2\"], [])", false, null},
                {"containsOnly([\"item1\"], [\"item1\", \"item2\"])", true, null},
                {"containsOnly([\"item1\", \"item1\"], [\"item1\", \"item2\"])", true, null},
                {"containsOnly([\"item1\", \"item2\"], [\"item1\", \"item2\"])", true, null},
                {"containsOnly([\"item1\", \"item2\"], [\"item2\", \"item1\"])", true, null},
                {"containsOnly(null, [])", null, ERROR},
                {"containsOnly([\"item1\", \"item2\"], null)", false, null},
                {"areElementsOf([\"item2\", \"item3\"], [\"item1\", \"item2\", \"item3\"])", true, null},
                {"areElementsOf([\"item0\", \"item3\"], [\"item1\", \"item2\", \"item3\"])", false, null},
                {"areElementsOf([], [\"item1\", \"item2\", \"item3\"])", true, null},
                {"areElementsOf(null, [\"item1\", \"item2\", \"item3\"])", true, null},
                {"areElementsOf([], null)", null, ERROR},
                {"zip([\"id\", \"value\"], [\"23a3e98\", \"c45da1b\"], [40, 120])", Arrays.asList(mapOf(entry("id", "23a3e98"), entry("value", new BigDecimal("40"))), mapOf(entry("id", "c45da1b"), entry("value", new BigDecimal("120")))), null},
                {"zip([\"id\", \"value\"], [[\"23a3e98\", \"c45da1b\"], [40, 120]])", Arrays.asList(mapOf(entry("id", "23a3e98"), entry("value", new BigDecimal("40"))), mapOf(entry("id", "c45da1b"), entry("value", new BigDecimal("120")))), null},
                {"zip([], [\"23a3e98\", \"c45da1b\"], [40, 120])", null, ERROR},
                {"zip([1], null)", null, ERROR},
                {"zip([\"a\"], [1], [2])", null, ERROR},
                {"zip([\"a\"], 1)", null, ERROR},
                {"zip([\"a\", \"b\"], [1, 2, 3], [2])", null, ERROR},
                {"remove([\"item1\", \"item2\"], \"item1\")", List.of("item2"), null},
                {"remove(null, \"item1\")", null, ERROR},
                {"remove([\"item1\", \"item2\"], null)", null, ERROR},
                {"removeAll([\"item1\", \"item2\", \"item3\"], [\"item1\", \"item2\"])", List.of("item3"), null},
                {"removeAll([\"item1\", \"item2\", \"item3\"], [\"item1\", \"item0\"])", Arrays.asList("item2", "item3"), null},
                {"removeAll(null, [\"item1\", \"item0\"])", null, ERROR},
                {"avg([3,5])", new BigDecimal("4"), null},
                {"avg([3,4,5])", new BigDecimal("4"), null},
                {"avg([4,5])", new BigDecimal("4.5"), null},
                {"avg(4)", new BigDecimal("4"), null},
                {"avg(n: null)", null, ERROR},
                {"max([5, 4, 10])", new BigDecimal("10"), null},
                {"min([5, 4, 10])", new BigDecimal("4"), null},
                {"median([2, 5, 10, 12, 34, 35])", new BigDecimal("11"), null},
                {"median([2, 5, 11, 12, 34, 35])", new BigDecimal("11.5"), null},
                {"median([5, 10, 12, 34, 35])", new BigDecimal("12"), null},
                {"median( null )", null, ERROR},
                {"median( [] )", null, ERROR},
                {"median( [\"a\", \"b\", \"c\"] )", null, ERROR},
                {"median( [\"a\", \"b\"] )", null, ERROR},
                {"mode([1, 2, 4, 4, 5, 6])", new BigDecimal("4"), null},
                {"mode([1, 1, 2, 4, 4, 5, 6])", Arrays.asList(new BigDecimal("1"), new BigDecimal("4")), null},
                {"mode(null)", null, ERROR},
                {"mode([])", null, ERROR},
                {"endsWith(\"Hello World!\", \"!\")", true, null},
                {"endsWith(\"Hello World!\", \"d\")", false, null},
                {"startsWith(\"Hello World!\", \"Hello\")", true, null},
                {"startsWith(\"Hello World!\", \"World\")", false, null},
                {"contains(\"Hello World!\", \"o World\")", true, null},
                // textOccurences is case sensitive - that's why only 6 occurences in this case
                {"textOccurrences(\"can\", \"Can you can a can as a canner can can a can?\")", new BigDecimal("6"), null},
                {"textOccurrences(\"can\", null)",null, ERROR},
                {"text(1, \"#.000\")", "1.000", null},
                {"text(123.456, \"#.000\")", "123.456", null},
                {"text(null, \"#.000\")", null, ERROR},
                {"text(1, null)", null, ERROR},
                {"text(123.456, \"#<invalid>$$%%\")", null, ERROR},
                {"left(\"Hello World!\", 5)", "Hello", null},
                {"left(null, 5)", null, ERROR},
                {"left(\"Hello World!\", null)", null, ERROR},
                {"right(\"Hello World!\", 6)", "World!", null},
                {"right(null, 6)", null, ERROR},
                {"right(\"Hello World!\", null)", null, ERROR},
                {"mid(\"Hello World!\", 7, 5)", "World", null},
                {"number(\"5\")", new BigDecimal("5"), null},
                {"number(\"5.5\")", new BigDecimal("5.5"), null},
                {"number(\"5,5\", 10)", new BigDecimal("10"), null},
                {"number(\"5.5\", 10)", new BigDecimal("5.5"), null},
                {"upper(\"lower\")", "LOWER", null},
                {"lower(\"UPPER\")", "upper", null},
                {"trim(\" Hello         World! \")", "Hello World!", null},
                {"trim(null)", null, ERROR},
                {"len(\"five\")", new BigDecimal("4"), null},
                {"isSpaces(\"     \")", true, null},
                {"isSpaces(\"   .  \")", false, null},
                {"isNumeric(\"2.3.5\")", false, null},
                {"isNumeric(\"2.3\")", true, null},
                {"isAlphanumeric(\"abcdefg5\")", true, null},
                {"isAlpha(\"abcdefg5\")", false, null},
                {"concat([\"a\", \"b\", \"c\"])", "abc", null},
                {"concat(\"a\", \"b\", \"c\")", "abc", null},
                {"concat(\"abc\")", "abc", null},
                {"concat([])", "", null},
                {"concat([\"a\", null, \"b\"])", null, ERROR},
                {"concat([1, 2, 3])", null, ERROR},
                {"concat(null)", null, ERROR},
                {"concat()", null, ERROR},
        };
        return Arrays.asList( cases );
    }
}
