/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.hop.databases.kingbasees;

import org.apache.hop.core.Const;
import org.apache.hop.core.database.BaseDatabaseMeta;
import org.apache.hop.core.database.DatabaseMeta;
import org.apache.hop.core.database.DatabaseMetaPlugin;
import org.apache.hop.core.database.IDatabase;
import org.apache.hop.core.gui.plugin.GuiPlugin;
import org.apache.hop.core.row.IValueMeta;

/** Contains KingbaseES specific information through static final members */
@DatabaseMetaPlugin(
    type = "KINGBASEES",
    typeDescription = "KingbaseES",
    documentationUrl = "/database/databases/kingbasees.html")
@GuiPlugin(id = "GUI-KingbaseDatabaseMeta")
public class KingbaseESDatabaseMeta extends BaseDatabaseMeta implements IDatabase {

  public static final String CONST_ALTER_TABLE = "ALTER TABLE ";

  /**
   * @return The extra option separator in database URL for this platform
   */
  @Override
  public String getExtraOptionSeparator() {
    return "&";
  }

  /**
   * @return This indicator separates the normal URL from the options
   */
  @Override
  public String getExtraOptionIndicator() {
    return "?";
  }

  @Override
  public int[] getAccessTypeList() {
    return new int[] {DatabaseMeta.TYPE_ACCESS_NATIVE};
  }

  @Override
  public int getDefaultDatabasePort() {
    if (getAccessType() == DatabaseMeta.TYPE_ACCESS_NATIVE) {
      return 54321;
    }
    return -1;
  }

  @Override
  public String getDriverClass() {
    return "com.kingbase.Driver";
  }

  @Override
  public String getURL(String hostname, String port, String databaseName) {
    return "jdbc:kingbase://" + hostname + ":" + port + "/" + databaseName;
  }

  /**
   * Checks whether or not the command setFetchSize() is supported by the JDBC driver...
   *
   * @return true is setFetchSize() is supported!
   */
  @Override
  public boolean isFetchSizeSupported() {
    return true;
  }

  /**
   * @return true if the database supports bitmap indexes
   */
  @Override
  public boolean isSupportsBitmapIndex() {
    return false;
  }

  /**
   * @return true if the database supports synonyms
   */
  @Override
  public boolean isSupportsSynonyms() {
    return false;
  }

  @Override
  public boolean isSupportsSequences() {
    return true;
  }

  /** Kingbase only support the data type: serial, it is not a real autoInc */
  @Override
  public boolean isSupportsAutoInc() {
    return false;
  }

  @Override
  public String getLimitClause(int nrRows) {
    return " limit " + nrRows;
  }

  /**
   * Get the SQL to get the next value of a sequence.
   *
   * @param sequenceName The sequence name
   * @return the SQL to get the next value of a sequence.
   */
  @Override
  public String getSqlNextSequenceValue(String sequenceName) {
    return "SELECT nextval('" + sequenceName + "')";
  }

  /**
   * Get the SQL to get the next value of a sequence.
   *
   * @param sequenceName The sequence name
   * @return the SQL to get the next value of a sequence.
   */
  @Override
  public String getSqlCurrentSequenceValue(String sequenceName) {
    return "SELECT currval('" + sequenceName + "')";
  }

  /**
   * Check if a sequence exists.
   *
   * @param sequenceName The sequence to check
   * @return The SQL to get the name of the sequence back from the databases data dictionary
   */
  @Override
  public String getSqlSequenceExists(String sequenceName) {
    return "SELECT relname AS sequence_name FROM sys_class WHERE relname = '"
        + sequenceName.toLowerCase()
        + "'";
  }

  @Override
  public String getSqlListOfSequences() {
    return "SELECT relname AS sequence_name FROM sys_class";
  }

  /**
   * Generates the SQL statement to add a column to the specified table
   *
   * @param tableName The table to add
   * @param v The column defined as a value
   * @param tk the name of the technical key field
   * @param useAutoinc whether or not this field uses auto increment
   * @param pk the name of the primary key field
   * @param semicolon whether or not to add a semi-colon behind the statement.
   * @return the SQL statement to add a column to the specified table
   */
  @Override
  public String getAddColumnStatement(
      String tableName, IValueMeta v, String tk, boolean useAutoinc, String pk, boolean semicolon) {
    return CONST_ALTER_TABLE
        + tableName
        + " ADD COLUMN "
        + getFieldDefinition(v, tk, pk, useAutoinc, true, false);
  }

  /**
   * Generates the SQL statement to drop a column from the specified table
   *
   * @param tableName The table to add
   * @param v The column defined as a value
   * @param tk the name of the technical key field
   * @param useAutoinc whether or not this field uses auto increment
   * @param pk the name of the primary key field
   * @param semicolon whether or not to add a semi-colon behind the statement.
   * @return the SQL statement to drop a column from the specified table
   */
  @Override
  public String getDropColumnStatement(
      String tableName, IValueMeta v, String tk, boolean useAutoinc, String pk, boolean semicolon) {
    return CONST_ALTER_TABLE + tableName + " DROP COLUMN " + v.getName() + Const.CR;
  }

  /**
   * Generates the SQL statement to modify a column in the specified table
   *
   * @param tableName The table to add
   * @param v The column defined as a value
   * @param tk the name of the technical key field
   * @param useAutoinc whether or not this field uses auto increment
   * @param pk the name of the primary key field
   * @param semicolon whether or not to add a semi-colon behind the statement.
   * @return the SQL statement to modify a column in the specified table
   */
  @Override
  public String getModifyColumnStatement(
      String tableName, IValueMeta v, String tk, boolean useAutoinc, String pk, boolean semicolon) {
    String retval = "";
    retval +=
        CONST_ALTER_TABLE + tableName + " DROP COLUMN " + v.getName() + Const.CR + ";" + Const.CR;
    retval +=
        CONST_ALTER_TABLE
            + tableName
            + " ADD COLUMN "
            + getFieldDefinition(v, tk, pk, useAutoinc, true, false);
    return retval;
  }

  @Override
  public String getFieldDefinition(
      IValueMeta v, String tk, String pk, boolean useAutoinc, boolean addFieldName, boolean addCr) {
    String retval = "";

    String fieldname = v.getName();
    int length = v.getLength();
    int precision = v.getPrecision();

    if (addFieldName) {
      retval += fieldname + " ";
    }

    int type = v.getType();
    switch (type) {
      case IValueMeta.TYPE_TIMESTAMP, IValueMeta.TYPE_DATE:
        retval += "TIMESTAMP";
        break;
      case IValueMeta.TYPE_BOOLEAN:
        if (isSupportsBooleanDataType()) {
          retval += "BOOLEAN";
        } else {
          retval += "CHAR(1)";
        }
        break;
      case IValueMeta.TYPE_NUMBER, IValueMeta.TYPE_INTEGER, IValueMeta.TYPE_BIGNUMBER:
        if (fieldname.equalsIgnoreCase(tk)
            || // Technical key
            fieldname.equalsIgnoreCase(pk) // Primary key
        ) {
          retval += "BIGSERIAL";
        } else {
          if (length > 0) {
            if (precision > 0 || length > 18) {
              retval += "NUMERIC(" + length + ", " + precision + ")";
            } else {
              if (length > 9) {
                retval += "BIGINT";
              } else {
                if (length < 5) {
                  retval += "SMALLINT";
                } else {
                  retval += "INTEGER";
                }
              }
            }

          } else {
            retval += "DOUBLE PRECISION";
          }
        }
        break;
      case IValueMeta.TYPE_STRING:
        if (length < 1 || length >= DatabaseMeta.CLOB_LENGTH) {
          retval += "TEXT";
        } else {
          retval += "VARCHAR(" + length + ")";
        }
        break;
      default:
        retval += " UNKNOWN";
        break;
    }

    if (addCr) {
      retval += Const.CR;
    }

    return retval;
  }

  /**
   * @return The SQL on this database to get a list of stored procedures.
   */
  @Override
  public String getSqlListOfProcedures() {
    return "select proname "
        + "from sys_proc, sys_user "
        + "where sys_user.usesysid = sys_proc.proowner "
        + "and upper(sys_user.usename) = '"
        + getUsername().toUpperCase()
        + "'";
  }

  /*
   * (non-Javadoc)
   *
   * @see com.kingbase.ketl.core.database.IDatabase#getReservedWords()
   */
  @Override
  public String[] getReservedWords() {
    return new String[] {
      // http://www.postgresql.org/docs/8.1/static/sql-keywords-appendix.html
      // added also non-reserved key words because there is progress from the Postgre developers to
      // add them
      "A",
      "ABORT",
      "ABS",
      "ABSOLUTE",
      "ACCESS",
      "ACTION",
      "ADA",
      "ADD",
      "ADMIN",
      "AFTER",
      "AGGREGATE",
      "ALIAS",
      "ALL",
      "ALLOCATE",
      "ALSO",
      "ALTER",
      "ALWAYS",
      "ANALYSE",
      "ANALYZE",
      "AND",
      "ANY",
      "ARE",
      "ARRAY",
      "AS",
      "ASC",
      "ASENSITIVE",
      "ASSERTION",
      "ASSIGNMENT",
      "ASYMMETRIC",
      "AT",
      "ATOMIC",
      "ATTRIBUTE",
      "ATTRIBUTES",
      "AUTHORIZATION",
      "AVG",
      "BACKWARD",
      "BEFORE",
      "BEGIN",
      "BERNOULLI",
      "BETWEEN",
      "BIGINT",
      "BINARY",
      "BIT",
      "BITVAR",
      "BIT_LENGTH",
      "BLOB",
      "BOOLEAN",
      "BOTH",
      "BREADTH",
      "BY",
      "C",
      "CACHE",
      "CALL",
      "CALLED",
      "CARDINALITY",
      "CASCADE",
      "CASCADED",
      "CASE",
      "CAST",
      "CATALOG",
      "CATALOG_NAME",
      "CEIL",
      "CEILING",
      "CHAIN",
      "CHAR",
      "CHARACTER",
      "CHARACTERISTICS",
      "CHARACTERS",
      "CHARACTER_LENGTH",
      "CHARACTER_SET_CATALOG",
      "CHARACTER_SET_NAME",
      "CHARACTER_SET_SCHEMA",
      "CHAR_LENGTH",
      "CHECK",
      "CHECKED",
      "CHECKPOINT",
      "CLASS",
      "CLASS_ORIGIN",
      "CLOB",
      "CLOSE",
      "CLUSTER",
      "COALESCE",
      "COBOL",
      "COLLATE",
      "COLLATION",
      "COLLATION_CATALOG",
      "COLLATION_NAME",
      "COLLATION_SCHEMA",
      "COLLECT",
      "COLUMN",
      "COLUMN_NAME",
      "COMMAND_FUNCTION",
      "COMMAND_FUNCTION_CODE",
      "COMMENT",
      "COMMIT",
      "COMMITTED",
      "COMPLETION",
      "CONDITION",
      "CONDITION_NUMBER",
      "CONNECT",
      "CONNECTION",
      "CONNECTION_NAME",
      "CONSTRAINT",
      "CONSTRAINTS",
      "CONSTRAINT_CATALOG",
      "CONSTRAINT_NAME",
      "CONSTRAINT_SCHEMA",
      "CONSTRUCTOR",
      "CONTAINS",
      "CONTINUE",
      "CONVERSION",
      "CONVERT",
      "COPY",
      "CORR",
      "CORRESPONDING",
      "COUNT",
      "COVAR_POP",
      "COVAR_SAMP",
      "CREATE",
      "CREATEDB",
      "CREATEROLE",
      "CREATEUSER",
      "CROSS",
      "CSV",
      "CUBE",
      "CUME_DIST",
      "CURRENT",
      "CURRENT_DATE",
      "CURRENT_DEFAULT_TRANSFORM_GROUP",
      "CURRENT_PATH",
      "CURRENT_ROLE",
      "CURRENT_TIME",
      "CURRENT_TIMESTAMP",
      "CURRENT_TRANSFORM_GROUP_FOR_TYPE",
      "CURRENT_USER",
      "CURSOR",
      "CURSOR_NAME",
      "CYCLE",
      "DATA",
      "DATABASE",
      "DATE",
      "DATETIME_INTERVAL_CODE",
      "DATETIME_INTERVAL_PRECISION",
      "DAY",
      "DEALLOCATE",
      "DEC",
      "DECIMAL",
      "DECLARE",
      "DEFAULT",
      "DEFAULTS",
      "DEFERRABLE",
      "DEFERRED",
      "DEFINED",
      "DEFINER",
      "DEGREE",
      "DELETE",
      "DELIMITER",
      "DELIMITERS",
      "DENSE_RANK",
      "DEPTH",
      "DEREF",
      "DERIVED",
      "DESC",
      "DESCRIBE",
      "DESCRIPTOR",
      "DESTROY",
      "DESTRUCTOR",
      "DETERMINISTIC",
      "DIAGNOSTICS",
      "DICTIONARY",
      "DISABLE",
      "DISCONNECT",
      "DISPATCH",
      "DISTINCT",
      "DO",
      "DOMAIN",
      "DOUBLE",
      "DROP",
      "DYNAMIC",
      "DYNAMIC_FUNCTION",
      "DYNAMIC_FUNCTION_CODE",
      "EACH",
      "ELEMENT",
      "ELSE",
      "ENABLE",
      "ENCODING",
      "ENCRYPTED",
      "END",
      "END-EXEC",
      "EQUALS",
      "ESCAPE",
      "EVERY",
      "EXCEPT",
      "EXCEPTION",
      "EXCLUDE",
      "EXCLUDING",
      "EXCLUSIVE",
      "EXEC",
      "EXECUTE",
      "EXISTING",
      "EXISTS",
      "EXP",
      "EXPLAIN",
      "EXTERNAL",
      "EXTRACT",
      "FALSE",
      "FETCH",
      "FILTER",
      "FINAL",
      "FIRST",
      "FLOAT",
      "FLOOR",
      "FOLLOWING",
      "FOR",
      "FORCE",
      "FOREIGN",
      "FORTRAN",
      "FORWARD",
      "FOUND",
      "FREE",
      "FREEZE",
      "FROM",
      "FULL",
      "FUNCTION",
      "FUSION",
      "G",
      "GENERAL",
      "GENERATED",
      "GET",
      "GLOBAL",
      "GO",
      "GOTO",
      "GRANT",
      "GRANTED",
      "GREATEST",
      "GROUP",
      "GROUPING",
      "HANDLER",
      "HAVING",
      "HEADER",
      "HIERARCHY",
      "HOLD",
      "HOST",
      "HOUR",
      "IDENTITY",
      "IGNORE",
      "ILIKE",
      "IMMEDIATE",
      "IMMUTABLE",
      "IMPLEMENTATION",
      "IMPLICIT",
      "IN",
      "INCLUDING",
      "INCREMENT",
      "INDEX",
      "INDICATOR",
      "INFIX",
      "INHERIT",
      "INHERITS",
      "INITIALIZE",
      "INITIALLY",
      "INNER",
      "INOUT",
      "INPUT",
      "INSENSITIVE",
      "INSERT",
      "INSTANCE",
      "INSTANTIABLE",
      "INSTEAD",
      "INT",
      "INTEGER",
      "INTERSECT",
      "INTERSECTION",
      "INTERVAL",
      "INTO",
      "INVOKER",
      "IS",
      "ISNULL",
      "ISOLATION",
      "ITERATE",
      "JOIN",
      "K",
      "KEY",
      "KEY_MEMBER",
      "KEY_TYPE",
      "LANCOMPILER",
      "LANGUAGE",
      "LARGE",
      "LAST",
      "LATERAL",
      "LEADING",
      "LEAST",
      "LEFT",
      "LENGTH",
      "LESS",
      "LEVEL",
      "LIKE",
      "LIMIT",
      "LISTEN",
      "LN",
      "LOAD",
      "LOCAL",
      "LOCALTIME",
      "LOCALTIMESTAMP",
      "LOCATION",
      "LOCATOR",
      "LOCK",
      "LOGIN",
      "LOWER",
      "M",
      "MAP",
      "MATCH",
      "MATCHED",
      "MAX",
      "MAXVALUE",
      "MEMBER",
      "MERGE",
      "MESSAGE_LENGTH",
      "MESSAGE_OCTET_LENGTH",
      "MESSAGE_TEXT",
      "METHOD",
      "MIN",
      "MINUTE",
      "MINVALUE",
      "MOD",
      "MODE",
      "MODIFIES",
      "MODIFY",
      "MODULE",
      "MONTH",
      "MORE",
      "MOVE",
      "MULTISET",
      "MUMPS",
      "NAME",
      "NAMES",
      "NATIONAL",
      "NATURAL",
      "NCHAR",
      "NCLOB",
      "NESTING",
      "NEW",
      "NEXT",
      "NO",
      "NOCREATEDB",
      "NOCREATEROLE",
      "NOCREATEUSER",
      "NOINHERIT",
      "NOLOGIN",
      "NONE",
      "NORMALIZE",
      "NORMALIZED",
      "NOSUPERUSER",
      "NOT",
      "NOTHING",
      "NOTIFY",
      "NOTNULL",
      "NOWAIT",
      "NULL",
      "NULLABLE",
      "NULLIF",
      "NULLS",
      "NUMBER",
      "NUMERIC",
      "OBJECT",
      "OCTETS",
      "OCTET_LENGTH",
      "OF",
      "OFF",
      "OFFSET",
      "OIDS",
      "OLD",
      "ON",
      "ONLY",
      "OPEN",
      "OPERATION",
      "OPERATOR",
      "OPTION",
      "OPTIONS",
      "OR",
      "ORDER",
      "ORDERING",
      "ORDINALITY",
      "OTHERS",
      "OUT",
      "OUTER",
      "OUTPUT",
      "OVER",
      "OVERLAPS",
      "OVERLAY",
      "OVERRIDING",
      "OWNER",
      "PAD",
      "PARAMETER",
      "PARAMETERS",
      "PARAMETER_MODE",
      "PARAMETER_NAME",
      "PARAMETER_ORDINAL_POSITION",
      "PARAMETER_SPECIFIC_CATALOG",
      "PARAMETER_SPECIFIC_NAME",
      "PARAMETER_SPECIFIC_SCHEMA",
      "PARTIAL",
      "PARTITION",
      "PASCAL",
      "PASSWORD",
      "PATH",
      "PERCENTILE_CONT",
      "PERCENTILE_DISC",
      "PERCENT_RANK",
      "PLACING",
      "PLI",
      "POSITION",
      "POSTFIX",
      "POWER",
      "PRECEDING",
      "PRECISION",
      "PREFIX",
      "PREORDER",
      "PREPARE",
      "PREPARED",
      "PRESERVE",
      "PRIMARY",
      "PRIOR",
      "PRIVILEGES",
      "PROCEDURAL",
      "PROCEDURE",
      "PUBLIC",
      "QUOTE",
      "RANGE",
      "RANK",
      "READ",
      "READS",
      "REAL",
      "RECHECK",
      "RECURSIVE",
      "REF",
      "REFERENCES",
      "REFERENCING",
      "REGR_AVGX",
      "REGR_AVGY",
      "REGR_COUNT",
      "REGR_INTERCEPT",
      "REGR_R2",
      "REGR_SLOPE",
      "REGR_SXX",
      "REGR_SXY",
      "REGR_SYY",
      "REINDEX",
      "RELATIVE",
      "RELEASE",
      "RENAME",
      "REPEATABLE",
      "REPLACE",
      "RESET",
      "RESTART",
      "RESTRICT",
      "RESULT",
      "RETURN",
      "RETURNED_CARDINALITY",
      "RETURNED_LENGTH",
      "RETURNED_OCTET_LENGTH",
      "RETURNED_SQLSTATE",
      "RETURNS",
      "REVOKE",
      "RIGHT",
      "ROLE",
      "ROLLBACK",
      "ROLLUP",
      "ROUTINE",
      "ROUTINE_CATALOG",
      "ROUTINE_NAME",
      "ROUTINE_SCHEMA",
      "ROW",
      "ROWS",
      "ROW_COUNT",
      "ROW_NUMBER",
      "RULE",
      "SAVEPOINT",
      "SCALE",
      "SCHEMA",
      "SCHEMA_NAME",
      "SCOPE",
      "SCOPE_CATALOG",
      "SCOPE_NAME",
      "SCOPE_SCHEMA",
      "SCROLL",
      "SEARCH",
      "SECOND",
      "SECTION",
      "SECURITY",
      "SELECT",
      "SELF",
      "SENSITIVE",
      "SEQUENCE",
      "SERIALIZABLE",
      "SERVER_NAME",
      "SESSION",
      "SESSION_USER",
      "SET",
      "SETOF",
      "SETS",
      "SHARE",
      "SHOW",
      "SIMILAR",
      "SIMPLE",
      "SIZE",
      "SMALLINT",
      "SOME",
      "SOURCE",
      "SPACE",
      "SPECIFIC",
      "SPECIFICTYPE",
      "SPECIFIC_NAME",
      "SQL",
      "SQLCODE",
      "SQLERROR",
      "SQLEXCEPTION",
      "SQLSTATE",
      "SQLWARNING",
      "SQRT",
      "STABLE",
      "START",
      "STATE",
      "STATEMENT",
      "STATIC",
      "STATISTICS",
      "STDDEV_POP",
      "STDDEV_SAMP",
      "STDIN",
      "STDOUT",
      "STORAGE",
      "STRICT",
      "STRUCTURE",
      "STYLE",
      "SUBCLASS_ORIGIN",
      "SUBLIST",
      "SUBMULTISET",
      "SUBSTRING",
      "SUM",
      "SUPERUSER",
      "SYMMETRIC",
      "SYSID",
      "SYSTEM",
      "SYSTEM_USER",
      "TABLE",
      "TABLESAMPLE",
      "TABLESPACE",
      "TABLE_NAME",
      "TEMP",
      "TEMPLATE",
      "TEMPORARY",
      "TERMINATE",
      "THAN",
      "THEN",
      "TIES",
      "TIME",
      "TIMESTAMP",
      "TIMEZONE_HOUR",
      "TIMEZONE_MINUTE",
      "TO",
      "TOAST",
      "TOP_LEVEL_COUNT",
      "TRAILING",
      "TRANSACTION",
      "TRANSACTIONS_COMMITTED",
      "TRANSACTIONS_ROLLED_BACK",
      "TRANSACTION_ACTIVE",
      "TRANSFORM",
      "TRANSFORMS",
      "TRANSLATE",
      "TRANSLATION",
      "TREAT",
      "TRIGGER",
      "TRIGGER_CATALOG",
      "TRIGGER_NAME",
      "TRIGGER_SCHEMA",
      "TRIM",
      "TRUE",
      "TRUNCATE",
      "TRUSTED",
      "TYPE",
      "UESCAPE",
      "UNBOUNDED",
      "UNCOMMITTED",
      "UNDER",
      "UNENCRYPTED",
      "UNION",
      "UNIQUE",
      "UNKNOWN",
      "UNLISTEN",
      "UNNAMED",
      "UNNEST",
      "UNTIL",
      "UPDATE",
      "UPPER",
      "USAGE",
      "USER",
      "USER_DEFINED_TYPE_CATALOG",
      "USER_DEFINED_TYPE_CODE",
      "USER_DEFINED_TYPE_NAME",
      "USER_DEFINED_TYPE_SCHEMA",
      "USING",
      "VACUUM",
      "VALID",
      "VALIDATOR",
      "VALUE",
      "VALUES",
      "VARCHAR",
      "VARIABLE",
      "VARYING",
      "VAR_POP",
      "VAR_SAMP",
      "VERBOSE",
      "VIEW",
      "VOLATILE",
      "WHEN",
      "WHENEVER",
      "WHERE",
      "WIDTH_BUCKET",
      "WINDOW",
      "WITH",
      "WITHIN",
      "WITHOUT",
      "WORK",
      "WRITE",
      "YEAR",
      "ZONE"
    };
  }

  /**
   * @param tableNames The names of the tables to lock
   * @return The SQL commands to lock database tables for write purposes.
   */
  @Override
  public String getSqlLockTables(String[] tableNames) {
    String sql = "LOCK TABLE ";
    for (int i = 0; i < tableNames.length; i++) {
      if (i > 0) {
        sql += ", ";
      }
      sql += tableNames[i] + " ";
    }
    sql += "IN ACCESS EXCLUSIVE MODE;" + Const.CR;

    return sql;
  }

  /**
   * @param tableName The name of the table to unlock
   * @return The SQL command to unlock a database table.
   */
  @Override
  public String getSqlUnlockTables(String[] tableName) {
    return null; // commit unlocks everything!
  }

  /**
   * @return true if the database defaults to naming tables and fields in uppercase. True for most
   *     databases except for stuborn stuff like Postgres ;-)
   */
  @Override
  public boolean isDefaultingToUppercase() {
    return false;
  }

  /**
   * @return true if the database supports timestamp to date conversion. Kingbase doesn't support
   *     this!
   */
  @Override
  public boolean isSupportsTimeStampToDateConversion() {
    return false;
  }

  @Override
  public void addDefaultOptions() {
    setSupportsBooleanDataType(true);
  }
}
