// Licensed to the Apache Software Foundation (ASF) under one
// or more contributor license agreements.  See the NOTICE file
// distributed with this work for additional information
// regarding copyright ownership.  The ASF licenses this file
// to you under the Apache License, Version 2.0 (the
// "License"); you may not use this file except in compliance
// with the License.  You may obtain a copy of the License at
//
//   http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing,
// software distributed under the License is distributed on an
// "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
// KIND, either express or implied.  See the License for the
// specific language governing permissions and limitations
// under the License.

package org.apache.doris.nereids.trees.expressions.functions.scalar;

import org.apache.doris.catalog.FunctionSignature;
import org.apache.doris.nereids.exceptions.AnalysisException;
import org.apache.doris.nereids.trees.expressions.Expression;
import org.apache.doris.nereids.trees.expressions.functions.AlwaysNotNullable;
import org.apache.doris.nereids.trees.expressions.functions.ComputeSignature;
import org.apache.doris.nereids.trees.expressions.functions.ImplicitlyCastableSignature;
import org.apache.doris.nereids.trees.expressions.literal.TinyIntLiteral;
import org.apache.doris.nereids.trees.expressions.shape.LeafExpression;
import org.apache.doris.nereids.trees.expressions.visitor.ExpressionVisitor;
import org.apache.doris.nereids.types.TimeV2Type;
import org.apache.doris.nereids.types.TinyIntType;

import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableList;

import java.util.List;

/**
 * ScalarFunction 'current_time'. This class is generated by GenerateFunction.
 */
public class CurrentTime extends ScalarFunction
        implements LeafExpression, ImplicitlyCastableSignature, AlwaysNotNullable, ComputeSignature {

    public static final List<FunctionSignature> SIGNATURES = ImmutableList.of(
            FunctionSignature.ret(TimeV2Type.INSTANCE).args(),
            FunctionSignature.ret(TimeV2Type.INSTANCE).args(TinyIntType.INSTANCE)
    );

    /**
     * constructor with 0 argument.
     */
    public CurrentTime() {
        super("current_time");
    }

    /**
     * constructor with 1 argument.
     */
    public CurrentTime(Expression arg) {
        super("current_time", arg);
    }

    /** constructor for withChildren and reuse signature */
    private CurrentTime(ScalarFunctionParams functionParams) {
        super(functionParams);
    }

    @Override
    public FunctionSignature computeSignature(FunctionSignature signature) {
        if (arity() == 1 && child(0) instanceof TinyIntLiteral) {
            byte precision = ((TinyIntLiteral) child(0)).getValue();
            if (precision < 0 || precision > 6) {
                throw new IllegalArgumentException("The precision must be between 0 and 6");
            }
            return FunctionSignature.ret(TimeV2Type.of(precision)).args(TinyIntType.INSTANCE);
        }

        return super.computeSignature(signature);
    }

    @Override
    public Expression withChildren(List<Expression> children) {
        Preconditions.checkArgument(children.isEmpty() || children.size() == 1);
        return new CurrentTime(getFunctionParams(children));
    }

    @Override
    public void checkLegalityAfterRewrite() {
        if (arity() == 1) {
            if (!child(0).isLiteral()) {
                throw new AnalysisException("CURTIME only accepts literal as precision.");
            }
        }
    }

    @Override
    public void checkLegalityBeforeTypeCoercion() {
        checkLegalityAfterRewrite();
    }

    @Override
    public List<FunctionSignature> getSignatures() {
        return SIGNATURES;
    }

    @Override
    public <R, C> R accept(ExpressionVisitor<R, C> visitor, C context) {
        return visitor.visitCurrentTime(this, context);
    }

    @Override
    public boolean isDeterministic() {
        return false;
    }
}
